<?php
ob_start();
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/functions.php';

// Check if user has permission to manage suppliers
if (!hasPermission('manage_suppliers')) {
    echo json_encode(['success' => false, 'message' => 'Permission denied']);
    exit;
}

// Check if the request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

$action = $_POST['action'] ?? '';

if ($action === 'approve_all') {
    $supplier_id = intval($_POST['supplier_id'] ?? 0);
    if ($supplier_id <= 0) {
        echo json_encode(['success' => false, 'message' => 'Invalid supplier ID']);
        exit;
    }
    
    try {
        $pdo->beginTransaction();
        
        // Update all pending receipts for the supplier to 'approved'
        $stmt = $pdo->prepare("UPDATE supplier_receipts SET status = 'approved' WHERE supplier_id = ? AND status = 'pending'");
        $stmt->execute([$supplier_id]);
        
        $pdo->commit();
        echo json_encode(['success' => true, 'message' => 'All pending receipts approved successfully']);
        exit;
    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Error approving receipts: ' . $e->getMessage()]);
        exit;
    }
} elseif ($action === 'update_status') {
    $receipt_id = intval($_POST['receipt_id'] ?? 0);
    $status = $_POST['status'] ?? '';
    
    if ($receipt_id <= 0 || !in_array($status, ['approved', 'rejected'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid receipt ID or status']);
        exit;
    }
    
    try {
        $pdo->beginTransaction();
        
        // Update the receipt status
        $stmt = $pdo->prepare("UPDATE supplier_receipts SET status = ? WHERE id = ?");
        $stmt->execute([$status, $receipt_id]);
        
        $pdo->commit();
        echo json_encode(['success' => true, 'message' => 'Receipt status updated successfully']);
        exit;
    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Error updating receipt status: ' . $e->getMessage()]);
        exit;
    }
} else {
    echo json_encode(['success' => false, 'message' => 'Invalid action']);
    exit;
}

$output = ob_get_clean();
if (!empty($output)) {
    echo json_encode(['success' => false, 'message' => 'Unexpected output: ' . $output]);
    exit;
}

?>

<style>
    .modal-dialog {
        display: flex;
        align-items: center;
        justify-content: center;
        min-height: 100vh;
    }
</style>

<!-- Warning Modal -->
<div class="modal fade" id="warningModal" tabindex="-1" aria-labelledby="warningModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="warningModalLabel">Warning</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>You are about to approve supplier items. Are you sure you want to proceed?</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="confirmApprove">Yes, Approve</button>
            </div>
        </div>
    </div>
</div>

<script>
    // Existing JavaScript code...

    function approveAllItems(supplierId) {
        // Show Bootstrap modal for warning and confirmation
        const warningModal = new bootstrap.Modal(document.getElementById('warningModal'));
        warningModal.show();

        // Set up the confirm button to proceed with approval
        document.getElementById('confirmApprove').onclick = function() {
            warningModal.hide();
            fetch('update_supplier_receipts.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=approve_all&supplier_id=${supplierId}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Supplier Items Approved');
                    location.reload();
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while updating the items.');
            });
        };
    }

    // Existing JavaScript code...
</script> 